#include "Stepper28.h"
#include <avr/interrupt.h>

// ตารางลำดับสำหรับ half-step
const uint8_t Stepper28::SEQ_HALF[8][4] = {
  {1,0,0,0},{1,1,0,0},{0,1,0,0},{0,1,1,0},
  {0,0,1,0},{0,0,1,1},{0,0,0,1},{1,0,0,1}
};

// ตารางลำดับสำหรับ full-step
const uint8_t Stepper28::SEQ_FULL[4][4] = {
  {1,0,0,1},{1,1,0,0},{0,1,1,0},{0,0,1,1}
};

// อินสแตนซ์หลักสำหรับ ISR
Stepper28* Stepper28::_inst = nullptr;

// ---------------- Constructor ----------------
Stepper28::Stepper28(uint8_t in1, uint8_t in2, uint8_t in3, uint8_t in4, step_mode_t mode)
  : _in1(in1), _in2(in2), _in3(in3), _in4(in4), _mode(mode) {
  _inst = this;                                       // เก็บตัวชี้อินสแตนซ์ปัจจุบัน
  _stepsPerRev = (_mode == HALF_STEP) ? 4096 : 2048;  // กำหนดจำนวนสเต็ปต่อรอบ
  _seqLen = (_mode == HALF_STEP) ? 8 : 4;             // ความยาวของตารางเฟส
}

// ---------------- begin ----------------
void Stepper28::begin() {
  pinMode(_in1, OUTPUT); pinMode(_in2, OUTPUT);
  pinMode(_in3, OUTPUT); pinMode(_in4, OUTPUT);
  stop();                                             // ตั้งค่าขาให้อยู่สถานะ LOW
  applyTimerFromRPM();                                // เตรียมค่า timer ตาม rpm
}

// ---------------- setSpeed ----------------
void Stepper28::setSpeed(int rpm) {
  if (rpm <= 0) return;
  _rpm = rpm;                                         // เก็บค่า rpm ใหม่
  applyTimerFromRPM();                                // คำนวณ timer ใหม่ตาม rpm
}

// ---------------- stop ----------------
void Stepper28::stop() {
  cli();                                              // ปิด interrupt ชั่วคราว
  _dir = 0;                                           // หยุดทิศทาง
  _targetActive = false;                              // ยกเลิกเป้าหมาย
  disableTimer();                                     // ปิด Timer
  digitalWrite(_in1, LOW); digitalWrite(_in2, LOW);
  digitalWrite(_in3, LOW); digitalWrite(_in4, LOW);   // ดับทุกขา
  sei();                                              // เปิด interrupt กลับ
}

// ---------------- stepCW / stepCCW ----------------
void Stepper28::stepCW(float value) {
  long steps = (fabs(value) < 10.0f) ? (long)(value * _stepsPerRev) : (long)value;  // แปลงรอบเป็นสเต็ป
  if (steps == 0) return;
  moveSteps(steps);                                  // สั่งหมุนตามจำนวนที่ต้องการ
}

void Stepper28::stepCCW(float value) {
  long steps = (fabs(value) < 10.0f) ? (long)(value * _stepsPerRev) : (long)value;
  if (steps == 0) return;
  moveSteps(-steps);                                 // หมุนทวนเข็ม
}

// ---------------- spinCW / spinCCW ----------------
void Stepper28::spinCW(int rpm) {
  setSpeed(rpm);                                     // ตั้งความเร็ว
  cli(); _dir = 1; _targetActive = false; enableTimer(); sei();  // เปิด timer หมุนต่อเนื่องตามเข็ม
}

void Stepper28::spinCCW(int rpm) {
  setSpeed(rpm);
  cli(); _dir = -1; _targetActive = false; enableTimer(); sei(); // เปิด timer หมุนต่อเนื่องทวนเข็ม
}

// ---------------- moveSteps / goTo ----------------
void Stepper28::moveSteps(long steps) { moveSteps(steps, _rpm); }

void Stepper28::moveSteps(long steps, int rpm) {
  if (steps == 0) return;
  cli();
  if (rpm > 0) setSpeed(rpm);
  _targetPos = _pos + steps;                         // ตั้งเป้าหมายตำแหน่ง
  _targetActive = true;                              // เปิดโหมด move-to
  _dir = (steps > 0) ? 1 : -1;                       // กำหนดทิศทาง
  enableTimer();                                     // เปิด Timer ให้เริ่มหมุน
  sei();
}

void Stepper28::goTo(long pos) { goTo(pos, _rpm); }

void Stepper28::goTo(long pos, int rpm) {
  long delta = pos - _pos; moveSteps(delta, rpm);     // คำนวณระยะที่ต้องหมุน
}

// ---------------- oneStep ----------------
void Stepper28::oneStep(int dir) {
  if (_mode == HALF_STEP) _seqPos = (_seqPos + (dir > 0 ? 1 : 7)) & 0x07;
  else _seqPos = (_seqPos + (dir > 0 ? 1 : 3)) & 0x03;
  writePhaseIdx(_seqPos);                            // ส่งสัญญาณขับขา
  _pos += dir;                                       // ปรับตำแหน่ง
  if (_targetActive && ((_dir > 0 && _pos >= _targetPos) || (_dir < 0 && _pos <= _targetPos))) {
    stop();                                          // ถึงเป้า → หยุด
  }
}

// ---------------- writePhaseIdx ----------------
void Stepper28::writePhaseIdx(uint32_t idx) {
  const uint8_t (*seq)[4] = (_mode == HALF_STEP) ? SEQ_HALF : SEQ_FULL;
  digitalWrite(_in1, seq[idx][0]); digitalWrite(_in2, seq[idx][1]);
  digitalWrite(_in3, seq[idx][2]); digitalWrite(_in4, seq[idx][3]);
}

// ---------------- applyTimerFromRPM ----------------
void Stepper28::applyTimerFromRPM() {
#if defined(__AVR__)
  const uint32_t prescaler = 8UL;
  double steps_per_min = (double)_rpm * (double)_stepsPerRev;
  double interval_sec = 60.0 / steps_per_min;
  double ticks_per_sec = (double)F_CPU / (double)prescaler;
  double ocr = interval_sec * ticks_per_sec - 1.0;
  if (ocr < 1.0) ocr = 1.0; if (ocr > 0xFFFF) ocr = 0xFFFF;
  cli(); _ocrValue = (uint32_t)round(ocr); sei();
#endif
}

// ---------------- enableTimer / disableTimer ----------------
void Stepper28::enableTimer() {
#if defined(__AVR__)
  cli(); TCCR1A = 0; TCCR1B = 0; TCNT1 = 0;
  TCCR1B |= (1 << WGM12); TCCR1B |= (1 << CS11);
  OCR1A = (uint16_t)_ocrValue; TIMSK1 |= (1 << OCIE1A); sei();
#endif
}

void Stepper28::disableTimer() {
#if defined(__AVR__)
  cli(); TIMSK1 &= ~(1 << OCIE1A); TCCR1B = 0; sei();
#endif
}

// ---------------- ISR ----------------
#if defined(__AVR__)
ISR(TIMER1_COMPA_vect) { Stepper28::isrHandler(); }
#endif

// ---------------- isrHandler ----------------
void Stepper28::isrHandler() {
  if (!_inst || _inst->_dir == 0) return;             // ถ้าไม่หมุนให้ข้าม
  _inst->oneStep(_inst->_dir);                        // ทำหนึ่งสเต็ปตามทิศทาง
}
